#############################################################################################
# Code for model selection via non-negative garrote separately for each asymmetry parameter #
#############################################################################################

rm(list=ls())

# load standard libraries
library(mgcv)
library(gtools)
library(MASS, quietly = T)
library(xtable)

# load expectile specific libraries
library(expectreg, quietly=T)
library(expectregSelection)
library(ExtraFunctions)

# load data, maps and additional functions
peru12s <- read.table('peru12_de_new.csv',header=T,sep=";",dec=",")
source("renameTables.R")
mapPeru2 <- read.bnd("Peru_borders.bnd")

# build neighbourhood structure
neighbours <- bnd2gra(mapPeru2)

# build factor variables
peru12s$meduC            <- as.factor(peru12s$meduC)
peru12s$edupartnerC      <- as.factor(peru12s$edupartnerC)
peru12s$deadchildrenD    <- as.factor(peru12s$deadchildrenD)
peru12s$householdmembers <- as.factor(peru12s$householdmembers)
peru12s$csex             <- as.factor(peru12s$csex)
peru12s$cbirthorder      <- as.factor(peru12s$cbirthorder)
peru12s$householdhead    <- as.factor(peru12s$householdhead)
peru12s$electricity      <- as.factor(peru12s$electricity)
peru12s$radio            <- as.factor(peru12s$radio)
peru12s$television       <- as.factor(peru12s$television)
peru12s$refrigerator     <- as.factor(peru12s$refrigerator)
peru12s$bicycle          <- as.factor(peru12s$bicycle)
peru12s$motorcycle       <- as.factor(peru12s$motorcycle)
peru12s$telephone        <- as.factor(peru12s$telephone)
peru12s$caesarian        <- as.factor(peru12s$caesarian)

levels(peru12s$csex) <- c("male","female")
levels(peru12s$meduC) <- c("low","medium","high")
levels(peru12s$edupartnerC) <- c("low","medium","high")

# look at the data
str(peru12s)
summary(peru12s)





set.seed(123456)

Sys.time()
cat(" \r\n   \r\n")

# Build intercept model as initial model for the selection
InterceptModel <- expectreg.ls(stunting~1, data=peru12s, expectiles=
                               c(0.02,0.05,0.1,0.2,0.5,0.8,0.9,0.95,0.98),
                               smooth="schall", lambda=1, estimate="laws", ci=F)

# Define maximal model
scopeSelection <- stunting ~ rb(cage) + 
    rb(breastfeeding) +
    rb(mbmi) + 
    rb(mheight) + 
    rb(mage) + 
    rb(mregion,type="markov",P=neighbours,bnd=mapPeru2) +
    csex + caesarian + cbirthorder +  
    meduC + edupartnerC + 
    householdmembers + householdhead + deadchildrenD +
    electricity + radio +television + refrigerator + bicycle + 
    motorcycle +  telephone

# Run model selection
SelectedModels <- garroteExpect(InterceptModel,scope=scopeSelection,trace=T,
                             grid_alpha=50, N_CV=10, delta_min=1e-10,grid_gamma=50,
                             output_type=c("list_models","one_model","list_models_delta","one_model_delta"),
                             type="separately",split="complete")
nameSim <- "garrote"
nameSim2 <- "Paper_c"

NameImg  <- paste("WS",      nameSim, nameSim2, ".RData", sep="_")
NameTable  <- paste("Table", nameSim, nameSim2, ".csv", sep="_")
NamePDF2 <- paste("Effects", nameSim, nameSim2, ".pdf"  , sep="_")

save.image(NameImg)
cat(" \r\n   \r\n")
Sys.time()
cat(" \r\n   \r\n")


# Build tables of selected models.
# Boosting has a different output => extra case
if(nameSim != "Boosting") {
    table1 <- SelectedModels$anova$table_selected
} else {
    coefList <- SelectedModels$coef
    SelectedMatrix <- matrix(0,ncol=ncol(coefList[[1]]),nrow=length(coefList))
    rownames(SelectedMatrix) <- names(coefList)
    colnames(SelectedMatrix) <- SelectedModels$asymmetries
    for(i in 1:length(coefList)) {
        for(j in 1:ncol(SelectedMatrix)) {
            if(any(coefList[[i]][,j] != 0)) {
                SelectedMatrix[i,j] <- 1
            }
        }
    }
    
    table0 <- renameTableBoosting(SelectedMatrix)
    table1 <- table0[-which(rownames(table0) == "bols(InterC, intercept = F)"          ),]
}

table2 <- table1[order_rows[order_rows %in% rownames(table1)],,drop=FALSE]
table3 <- renameTable(table2)
colnames(table3) <- gsub("X","",colnames(table3))

write.table(table3,NameTable,dec=".",sep=";")


# Plot resulting models
ylim_factor <- c(-0.57,0.57)    
ylim_spat   <- c(-0.55,0.75)

if(nameSim != "Boosting") {
    xlab_old <- names(SelectedModels$one_model$coef)
    xlab_new <- rename_xlab(xlab_old)
    type <- "one_model"
    if(!is.null(SelectedModels$one_model_delta)) {
        type <- "one_model_delta"
    }
    
    pdf(NamePDF2,width=7,height=7)
    par(mar=c(4,4,2,2))
    plot(SelectedModels,ask=F,ylim=c(-2,2),xlab=xlab_new,mar.min=2,
         legend=T,cex.main=2,cols="hcl",type=type,
         ylim_factor=ylim_factor,ylim_spat=ylim_spat)
    plot(SelectedModels,ask=F,ylim=c(-2,2),xlab=xlab_new,mar.min=2,
         legend=F,cex.main=2,cols="hcl",type=type,
         ylim_factor=ylim_factor,ylim_spat=ylim_spat)
    dev.off()
} else {
    pdf(NamePDF2,width=7,height=7)
    par(mar=c(4,4,2,2))
    plot(SelectedModels,ylim=c(-2,2))
    dev.off()
}    

