# Basic Example to explain the code of expectile regression with mgcv


Sys.getpid()

rm(list=ls())
gc(TRUE)
library(mgcv)
library(Matrix)
library(MASS)

source("function_expectreg_mgcv_bam11.R")

# Characteristics of the simulated data
n      <- 5000
n_pred <- 10000
N <- 1
Sim_type <- "1"
h_type <- "heterosc"

# Characteristics of the model
delta        <- 1e-4          # Tolerance in smoothing parameter estimation via schall (Differences in the exact smoothing parameters)
reltol       <- 1e-5          # Tolerance in smoothing parameter estimation via GCV    (Difference in gcv criterion)
initial_w    <- TRUE          # Shall the estimated weights of the initial expectile regression used as initial weights for further models
step_max     <- 10            # Maximal Number of iteration for step hlaving in Schall algorithm
bounce_off   <- TRUE          # Schall the optimization via gcv also have limitations for the possible smoothing parameters


# Formula of the model
f_exp <- y ~ ti(x1,k=15,bs="ps",np=F) + ti(x2,k=15,bs="ps",np=F) + ti(x1,x2,k=c(15,15),bs=c("ps","ps"),np=F)


# Simulation design
f1 <- function(x1,x2,sx1,sx2){
    10*pi*sx1*sx2*(1.2*exp(-(x1-0.2)^2/sx1^2 - (x2-0.3)^2/sx2^2) + 0.8*exp(-(x1-0.7)^2/sx1^2 - (x2-0.8)^2/sx2^2))
}

f2 <- function(x1,x2){
    2*sin(pi*x1) + exp(2*x2)
}

Sim_Data_fun <- function(n, Sim_type = c("1","2"), h_type = c("homosc","heterosc"),f1,f2) {
    x1 <- runif(n)
    x2 <- runif(n)
    
    sx1 <- 0.3
    sx2 <- 0.4
    
    if(Sim_type == "1") {
        eta <- f1(x1=x1,x2=x2,sx1=sx1,sx2=sx2)
    }
    if(Sim_type == "2") {
        eta <- f2(x1=x1,x2=x2)
    }
    
    if(h_type == "homosc") {
        eps <- rnorm(n, mean=0, sd=2)
    }
    if(h_type == "heterosc") {
        eps <- rnorm(n, mean=0, sd=(1/1.5*(x1+1) + 1/1.5*(x2+1)))
    }
    y <- eta + eps
    
    data.frame(y,x1,x2)
}

# Function to get the (P)MWSE
DEV <- function(y,fitted,expectile=0.5) {
    weights <- rep(expectile,times=length(y))
    weights[y >= fitted] <- expectile
    weights[y < fitted] <- 1 - expectile
    mean(weights*(y-fitted)^2)
}

# Simulate Data
data_list <- list()

set.seed(1)

for(i in 1:N) {
    data_list[[i]] <- Sim_Data_fun(n,Sim_type=Sim_type, h_type=h_type,f1=f1,f2=f2)        
}

data_pred <- Sim_Data_fun(n_pred,Sim_type=Sim_type, h_type=h_type,f1=f1,f2=f2) 

data1 <- data_list[[1]]

# Initial Model to get the initial smoothing parameters
set.seed(111)
model1 <- bam(f_exp, data = data1, family = gaussian(), weight=rep(0.5,times=n),method="GCV.Cp")


sm_par_vec2 <- model1$sp

quietly <- FALSE

# Expectile Regression with smoothing parameters estimated via generalized Fellner-Schall
set.seed(111)
model_10_bam_sch_d <- expectreg_bam_smooth(formula = f_exp, data = data1, expectiles = 0.10, 
                                           sm_par_vec = sm_par_vec2, fixed = FALSE,
                                           delta = delta, step_max = step_max, 
                                           reltol = reltol, initial_w = initial_w, 
                                           opt_type = "schall", dev_schall = "dev",
                                           quietly = quietly, bounce_off = bounce_off)

# Expectile Regression with smoothing parameters estimated via GCV
set.seed(111)
model_10_bam_gcv <- expectreg_bam_smooth(formula = f_exp, data = data1, expectiles = 0.10, 
                                         sm_par_vec = sm_par_vec2, fixed = FALSE,
                                         delta = delta, step_max = step_max, 
                                         reltol = reltol, initial_w = initial_w, 
                                         opt_type = "gcv", dev_schall = "non",
                                         quietly = quietly,bounce_off = bounce_off)



# Calculate (P)MWSE
fitted_10_bam_sch_d <- predict(model_10_bam_sch_d$model, newdata = data_pred, "response")
fitted_10_bam_gcv   <- predict(model_10_bam_gcv  $model, newdata = data_pred, "response")

MWSE_sch_d  <- DEV(data1$y    , model_10_bam_sch_d$fitted, 0.10)
PMWSE_sch_d <- DEV(data_pred$y, fitted_10_bam_sch_d      , 0.10)

MWSE_gcv    <- DEV(data1$y    , model_10_bam_gcv  $fitted, 0.10)
PMWSE_gcv   <- DEV(data_pred$y, fitted_10_bam_gcv        , 0.10)


