### Code for estimating spatio-temporal expectile regression
# Note: This model needs several days of runtime. 


rm(list=ls())
gc(TRUE)
Sys.getpid()

# Load packages
library(mgcv)
library(Matrix)
library(MASS)


# Set characteristics for the model
min_year     <- 1980          # Minimal year of the data
fixed_s      <- FALSE         # Fix smoothing parameters
use_year     <- TRUE          # Include year as covariate
expectile1   <- 0.10          # Asymmetry parameter
spline_type  <- "ps"          # Spline ps or tp for mgcv
isotropy     <- "aniso"       # Type of istrotropy, possibilities: "isotr" # "aniso" "joint"
delta        <- 1e-4          # Tolerance in smoothing parameter estimation via schall (Differences in the exact smoothing parameters)
reltol       <- 1e-5          # Tolerance in smoothing parameter estimation via GCV    (Difference in gcv criterion)
initial_w    <- TRUE          # Shall the estimated weights of the initial expectile regression used as initial weights for further models
step_max     <- 10            # Maximal Number of iteration for step hlaving in Schall algorithm
fitting_type <- "gcv"         # Esimation of smoothing parameters via "gcv" or generalized Fellner-Schall algorithm ("schall")
dev_schall   <- "gcv"         # Check for increasing model fit in schall based on gcv or penalized deviance ("gcv" vs "dev")
bounce_off   <- TRUE          # Schall the optimization via gcv also have limitations for the possible smoothing parameters

# Load expectile function
source("function_expectreg_mgcv_bam11.R")


# Build name for saving
fixed_s_text <- "fixed"
if(!fixed_s) {
    fixed_s_text <- "optim"
}

use_year_text <- "noyear"
if(use_year) {
    use_year_text <- "year"
}

delta_text <- as.character(delta)
delta_text <- gsub(x=delta_text,pattern=".",replacement="_",fixed=T)
delta_text <- paste(delta_text,step_max,sep="_")
reltol_text <- as.character(reltol)
reltol_text <- gsub(x=reltol_text,pattern=".",replacement="_",fixed=T)
reltol_text <- paste(reltol_text,step_max,sep="_")

expect_text <- as.character(expectile1*100)
initial_w_text <- "T"
if(!initial_w) initial_w_text <- "F"


NameSave <- paste("SpatTemp3b",min_year,use_year_text,spline_type,isotropy,fixed_s_text,expect_text,fitting_type,dev_schall,delta_text,reltol_text,initial_w_text,".RData",sep="_")

# Load data
homstart <- read.table("Table_Temperatur_new3b.txt",sep=";",dec=".",header=T)


homstart <- subset(homstart, year >= min_year)
homstart <- subset(homstart, elevation <= 1850)
homstart$raw[homstart$raw < -100] <- NA
homstart$y <- homstart$raw

homstart <- homstart[!is.na(homstart$y),]

homstart <- homstart[,c("y","elevation","day","lon","lat","year")]

# Build formula of the model
if(isotropy == "isotr") {
    if(spline_type != "tp") {
        stop ("Isotropy only for tp basis")
    }
    if(use_year){
            f_exp <- y ~ s(elevation, bs="tp", k=7) + s(year, bs="tp", k=15, pc=1980) + ti(day,bs="cc",k=15, np=F) + 
                ti(lon,lat,bs="tp",d=2,k=54, np=F) + ti(day,lon,lat,bs=c("cc","tp"),d=c(1,2),k=c(15,54), np=F)
        } else {
             f_exp <- y ~ s(elevation, bs="tp", k=7) + ti(day,bs="cc",k=15, np=F) + 
                ti(lon,lat,bs="tp",d=2,k=54, np=F) + ti(day,lon,lat,bs=c("cc","tp"),d=c(1,2),k=c(15,54), np=F)
    }
} 
if(isotropy == "aniso"){
    if(use_year){
        if(spline_type == "ps") {
            f_exp <- y ~ s(elevation, bs="ps", k=7) + s(year, bs="ps", k=15, pc=1980) + 
                ti(day, bs="cp",k=15, np=F) + ti(lon, bs="ps",k=6, np=F) + ti(lat, bs="ps",k=9, np=F) + 
                ti(lon,lat, bs=c("ps","ps"),d=c(1,1),k=c(6,9), np=F) + 
                ti(lon,day, bs=c("ps","cp"),d=c(1,1),k=c(6,15), np=F) + 
                ti(day,lat, bs=c("cp","ps"),d=c(1,1),k=c(15,9), np=F) +
                ti(day,lon,lat, bs=c("cp","ps","ps"),d=c(1,1,1),k=c(15,6,9), np=F)
        }
        if(spline_type == "tp") {
            f_exp <- y ~ s(elevation, bs="tp", k=7) + s(year, bs="tp", k=15, pc=1980) + 
                    ti(day, bs="cc",k=15, np=F) + ti(lon, bs="tp",k=6, np=F) + ti(lat, bs="tp",k=9, np=F) + 
                    ti(lon,lat, bs=c("tp","tp"),d=c(1,1),k=c(6,9), np=F) + 
                    ti(lon,day, bs=c("tp","cc"),d=c(1,1),k=c(6,15), np=F) + 
                    ti(day,lat, bs=c("cc","tp"),d=c(1,1),k=c(15,9), np=F) +
                    ti(day,lon,lat, bs=c("cc","tp","tp"),d=c(1,1,1),k=c(15,6,9), np=F)
        }
    } else {
        if(spline_type == "ps") {
            f_exp <- y ~ s(elevation, bs="ps", k=7) +  
                ti(day, bs="cp",k=15, np=F) + ti(lon, bs="ps",k=6, np=F) + ti(lat, bs="ps",k=9, np=F) + 
                ti(lon,lat, bs=c("ps","ps"),d=c(1,1),k=c(6,9), np=F) + 
                ti(lon,day, bs=c("ps","cp"),d=c(1,1),k=c(6,15), np=F) + 
                ti(day,lat, bs=c("cp","ps"),d=c(1,1),k=c(15,9), np=F) +
                ti(day,lon,lat, bs=c("cp","ps","ps"),d=c(1,1,1),k=c(15,6,9), np=F)
        }
        if(spline_type == "tp") {
            f_exp <- y ~ s(elevation, bs="tp", k=7) + 
                ti(day, bs="cc",k=15, np=F) + ti(lon, bs="tp",k=6, np=F) + ti(lat, bs="tp",k=9, np=F) + 
                ti(lon,lat, bs=c("tp","tp"),d=c(1,1),k=c(6,9), np=F) + 
                ti(lon,day, bs=c("tp","cc"),d=c(1,1),k=c(6,15), np=F) + 
                ti(day,lat, bs=c("cc","tp"),d=c(1,1),k=c(15,9), np=F) +
                ti(day,lon,lat, bs=c("cc","tp","tp"),d=c(1,1,1),k=c(15,6,9), np=F)
        }
    }
}

if(isotropy == "joint"){
    if(use_year){
        if(spline_type == "ps") {
            f_exp <- y ~ s(elevation, bs="ps", k=7) + s(year, bs="ps", k=15, pc=1980) + 
                te(day,lon,lat, bs=c("cp","ps","ps"),d=c(1,1,1),k=c(15,6,9), np=F)
        }
        if(spline_type == "tp") {
            f_exp <- y ~ s(elevation, bs="tp", k=7) + s(year, bs="tp", k=15, pc=1980) + 
                te(day,lon,lat, bs=c("cc","tp","tp"),d=c(1,1,1),k=c(15,6,9), np=F)
        }
    } else {
        if(spline_type == "ps") {
            f_exp <- y ~ s(elevation, bs="ps", k=7) +  
                te(day,lon,lat, bs=c("cp","ps","ps"),d=c(1,1,1),k=c(15,6,9), np=F)
        }
        if(spline_type == "tp") {
            f_exp <- y ~ s(elevation, bs="tp", k=7) + 
                te(day,lon,lat, bs=c("cc","tp","tp"),d=c(1,1,1),k=c(15,6,9), np=F)
        }
    }
}

# Get initial values for the smoothing parameters
Sys.time()
set.seed(111)
rainmodel1 <- bam(f_exp, data = homstart, family = gaussian(), weight=rep(0.5,times=nrow(homstart)),method="GCV.Cp")
print(object.size(rainmodel1),units="Mb")
Sys.time()
gc(TRUE)
save.image(NameSave)
Sys.time()

sm_par_vec2 <- rainmodel1$sp

# if(any(sm_par_vec2 > 1e5))  sm_par_vec2[sm_par_vec2 > 1e5]  <- 1e5
# if(any(sm_par_vec2 < 1e-5)) sm_par_vec2[sm_par_vec2 < 1e-5] <- 1e-5
print(sm_par_vec2)

Sys.time()

quietly <- FALSE

# Run expectile regression
Sys.time()
gc(TRUE)
save.image(NameSave)
set.seed(111)
rainmodel2 <- expectreg_bam_smooth(formula=f_exp, data=homstart, expectiles=expectile1, sm_par_vec = sm_par_vec2, fixed=fixed_s,
                                   delta=delta, step_max=step_max, reltol = reltol, initial_w = initial_w, opt_type = fitting_type, 
                                   dev_schall = dev_schall, quietly=quietly, bounce_off=bounce_off)

Sys.time()
gc(TRUE)

# Save the estimated model
save.image(NameSave)


